/***************************************************************************
 *   basecode - archive of useful C++ classes                              *
 *   Copyright (C) 2004 by Michal Turek - Woq                              *
 *   WOQ (at) seznam.cz, http://woq.nipax.cz/                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ccameraapp.h"

namespace basecode
{

CCameraApp::CCameraApp(int argc, char *argv[]) :
		CApplication(argc, argv),
		m_cam(CVector<float>(0.0f, 0.0f, 0.0f))
{
	m_bullets_pos.reserve(100);
	m_bullets_dir.reserve(100);
}

CCameraApp::~CCameraApp()
{

}

void CCameraApp::Init(const string& win_title)
{
	CApplication::Init(win_title);
	SDL_ShowCursor(SDL_DISABLE);
}

void CCameraApp::InitGL()
{
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glClearDepth(1.0);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_DEPTH_TEST);
	glShadeModel(GL_SMOOTH);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	glEnable(GL_LINE_SMOOTH);
	glLineWidth(3.0f);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void CCameraApp::OnInit()
{

}

void CCameraApp::Draw()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	m_cam.LookAt();

	glColor4ub(30, 200, 30, 255);
	CGrid::DrawPlaneXZ(SIZE, SIZE/4.0f, -0.2f);
	glColor4ub(255, 0, 0, 255);
	CGrid::DrawAxis(SIZE / 2.0f);

	glDisable(GL_BLEND);
	glBegin(GL_QUADS);
	glColor3ub(0, 0, 255);

	vector<CVector<float> >::iterator it;

	for(it = m_bullets_pos.begin(); it != m_bullets_pos.end(); it++)
	{
		glVertex3fv(*it + CVector<float>(-1,-1, 0));
		glVertex3fv(*it + CVector<float>( 1,-1, 0));
		glVertex3fv(*it + CVector<float>( 1, 1, 0));
		glVertex3fv(*it + CVector<float>(-1, 1, 0));
	}

	glEnd();
	glEnable(GL_BLEND);
}

void CCameraApp::Update()
{
	SDL_PumpEvents();

	Uint8* keys;
	keys = SDL_GetKeyState(NULL);

	if(keys[SDLK_UP] == SDL_PRESSED || keys[SDLK_w] == SDL_PRESSED)
		m_cam.GoFront(GetFPS());
	if(keys[SDLK_DOWN] == SDL_PRESSED || keys[SDLK_s] == SDL_PRESSED)
		m_cam.GoBack(GetFPS());
	if(keys[SDLK_LEFT] == SDL_PRESSED || keys[SDLK_a] == SDL_PRESSED)
		m_cam.GoLeft(GetFPS());
	if(keys[SDLK_RIGHT] == SDL_PRESSED || keys[SDLK_d] == SDL_PRESSED)
		m_cam.GoRight(GetFPS());

	m_cam.SetYPos(5.0f);


	vector<CVector<float> >::iterator it_pos;
	vector<CVector<float> >::iterator it_dir;

	for(it_pos = m_bullets_pos.begin(), it_dir = m_bullets_dir.begin();
	    it_pos != m_bullets_pos.end() || it_dir != m_bullets_dir.end();
	    it_pos++, it_dir++)
	{
		*it_pos += *it_dir / GetFPS();
	}
}

bool CCameraApp::ProcessEvent(SDL_Event& event)
{
	switch(event.type)
	{
	case SDL_MOUSEMOTION:
		// SDL_WarpMouse() generates SDL_MOUSEMOTION event :-(
		if(event.motion.x != GetWinWidth() >> 1
		|| event.motion.y != GetWinHeight() >> 1)
		{
			// First several messages MUST be ignored
			static int kriza = 0;
			if(kriza++ < 5)
				break;

			m_cam.Rotate(event.motion.xrel,
					event.motion.yrel, GetFPS());

			// Center mouse in window
			SDL_WarpMouse(GetWinWidth()>>1, GetWinHeight()>>1);
		}
		break;

	case SDL_MOUSEBUTTONDOWN:
		m_bullets_pos.push_back(CVector<float>(m_cam.GetPos()));
		m_bullets_dir.push_back(CVector<float>(m_cam.GetDir()*20.0f));
		break;


	default:// Other events
		return CApplication::ProcessEvent(event);
		break;
	}

	return true;
}

}
