/***************************************************************************
 *   basecode - archive of useful C++ classes                              *
 *   Copyright (C) 2004 by Michal Turek - Woq                              *
 *   WOQ (at) seznam.cz, http://woq.nipax.cz/                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef __CVECTOR_H__
#define __CVECTOR_H__

#include <cmath>
#include <iostream>
#include "basecode.h"

// Only private use
#define X m_data[0]
#define Y m_data[1]
#define Z m_data[2]

using namespace std;

namespace basecode
{

template <class TYPE>
class CVector
{
public:
	CVector();
	CVector(TYPE x, TYPE y, TYPE z);
	CVector(const CVector<TYPE>& v);

	// Vector from two points
	CVector(const CVector<TYPE>& start_point,
			const CVector<TYPE>& end_point);

	~CVector();


	void Set(TYPE x = 0, TYPE y = 0, TYPE z = 0);
	void SetX(TYPE x) { X = x; }
	void SetY(TYPE y) { Y = y; }
	void SetZ(TYPE z) { Z = z; }

	TYPE GetX() const { return X; }
	TYPE GetY() const { return Y; }
	TYPE GetZ() const { return Z; }


	// Return pointer to data (for glVertex3*v())
	TYPE* GetDataPtr() const { return (TYPE*)m_data; }
	operator TYPE*() const { return (TYPE*)m_data; }
	operator const TYPE*() const { return m_data; }

	bool IsZero() const { return (X == 0 && Y == 0 && Z == 0); }


	const CVector<TYPE>& operator=(const CVector<TYPE>& v);

	void operator+=(const CVector<TYPE>& v);
	void operator-=(const CVector<TYPE>& v);

	CVector<TYPE> operator+(const CVector<TYPE>& v) const;
	CVector<TYPE> operator-(const CVector<TYPE>& v) const;

	// Inverse vector
	CVector<TYPE> operator-() const { return CVector<TYPE>(-X, -Y, -Z); }


	// Vector scaling
	void operator*=(TYPE n);
	void operator/=(TYPE n);
	CVector<TYPE> operator*(TYPE n) const;
	CVector<TYPE> operator/(TYPE n) const;

	// Vectors can be MATHEMATICAL equal (use Normalize() before)
	bool operator==(const CVector<TYPE>& v) const;
	bool operator!=(const CVector<TYPE>& v) const;


	// Dot product
	TYPE operator*(const CVector<TYPE>& v) const;
	TYPE Dot(const CVector<TYPE>& v) const;


	// Cross product from two vectors
	CVector<TYPE> Cross(const CVector<TYPE>& v) const;

	// Cross product from three points
	CVector<TYPE> Cross(const CVector<TYPE>& p1,
			const CVector<TYPE>& p2) const;


	double Magnitude() const;
	void Normalize();

	double AngleRad(const CVector<TYPE>& v) const;
	double AngleDeg(const CVector<TYPE>& v) const;

	// Distance of two points
	double Distance(const CVector<TYPE>& v) const;

private:
	TYPE m_data[3];
};



/////////////////////////////////////////////////////////////////////////////// Implementation

template <class TYPE>
CVector<TYPE>::CVector()
{
	X = (TYPE)0;
	Y = (TYPE)0;
	Z = (TYPE)0;
}

template <class TYPE>
CVector<TYPE>::CVector(const CVector<TYPE>& v)
{
	X = v.X;
	Y = v.Y;
	Z = v.Z;
}

template <class TYPE>
CVector<TYPE>::CVector(const CVector<TYPE>& start_point,
			const CVector<TYPE>& end_point)
{
	*this = end_point - start_point;
}

template <class TYPE>
CVector<TYPE>::CVector(TYPE x, TYPE y, TYPE z)
{
	X = x;
	Y = y;
	Z = z;
}

template <class TYPE>
CVector<TYPE>::~CVector()
{

}

template <class TYPE>
inline void CVector<TYPE>::Set(TYPE x, TYPE y, TYPE z)
{
	X = x;
	Y = y;
	Z = z;
}

template <class TYPE>
inline void CVector<TYPE>::operator+=(const CVector<TYPE>& v)
{
	X += v.X;
	Y += v.Y;
	Z += v.Z;
}

template <class TYPE>
inline void CVector<TYPE>::operator-=(const CVector<TYPE>& v)
{
	X -= v.X;
	Y -= v.Y;
	Z -= v.Z;
}

template <class TYPE>
inline void CVector<TYPE>::operator*=(TYPE n)
{
	X *= n;
	Y *= n;
	Z *= n;
}

template <class TYPE>
inline void CVector<TYPE>::operator/=(TYPE n)
{
	if(n != 0)
	{
		X /= n;
		Y /= n;
		Z /= n;
	}
}

template <class TYPE>
inline const CVector<TYPE> & CVector<TYPE>::operator=(const CVector<TYPE>& v)
{
	X = v.X;
	Y = v.Y;
	Z = v.Z;
	return *this;
}

template <class TYPE>
inline CVector<TYPE> CVector<TYPE>::operator+(const CVector<TYPE>& v) const
{
	return CVector(X + v.X, Y + v.Y, Z + v.Z);
}

template <class TYPE>
inline CVector<TYPE> CVector<TYPE>::operator-(const CVector<TYPE>& v) const
{
	return CVector(X - v.X, Y - v.Y, Z - v.Z);
}

template <class TYPE>
inline CVector<TYPE> CVector<TYPE>::operator*(TYPE n) const
{
	return CVector(X * n, Y * n, Z * n);
}

template <class TYPE>
inline CVector<TYPE> CVector<TYPE>::operator/(TYPE n) const
{
	if(n != 0)
	{
		return CVector(X / n, Y / n, Z / n);
	}

	return *this;
}

// Dot product
template <class TYPE>
inline TYPE CVector<TYPE>::operator*(const CVector<TYPE>& v) const
{
	return X*v.X + Y*v.Y + Z*v.Z;
}

template <class TYPE>
inline bool CVector<TYPE>::operator==(const CVector<TYPE>& v) const
{
	// Vectors can be MATHEMATICAL equal!!! (use Normalize() before)
	 return (X == v.X && Y == v.Y && Z == v.Z);
}

template <class TYPE>
inline bool CVector<TYPE>::operator!=(const CVector<TYPE>& v) const
{
	return !(*this == v);
}

template <class TYPE>
inline double CVector<TYPE>::Magnitude() const
{
	return sqrt(X*X + Y*Y + Z*Z);
}

template <class TYPE>
inline void CVector<TYPE>::Normalize()
{
	*this /= this->Magnitude();
}

template <class TYPE>
inline TYPE CVector<TYPE>::Dot(const CVector<TYPE>& v) const
{
	return X*v.X + Y*v.Y + Z*v.Z;
}

// Cross product from two vectors
template <class TYPE>
CVector<TYPE> CVector<TYPE>::Cross(const CVector<TYPE>& v) const
{
	CVector ret;

	ret.X = Y * v.Z - Z * v.Y;
	ret.Y = Z * v.X - X * v.Z;
	ret.Z = X * v.Y - Y * v.X;

	return ret;
}

// Cross product from three points
template <class TYPE>
CVector<TYPE> CVector<TYPE>::Cross(const CVector<TYPE>& p1,
				const CVector<TYPE>& p2) const
{
	CVector<TYPE> v1(*this, p1);
	CVector<TYPE> v2(*this, p2);

	return v1.Cross(v2);
}

template <class TYPE>
inline double CVector<TYPE>::AngleRad(const CVector<TYPE>& v) const
{
	TYPE magnitudes = Magnitude() * v.Magnitude();

	if(magnitudes == 0)
		return 0;

	return acos((*this * v) / magnitudes);
}

template <class TYPE>
inline double CVector<TYPE>::AngleDeg(const CVector<TYPE>& v) const
{
	return RADTODEG(AngleRad(v));
}

// Distance of two points
template <class TYPE>
inline double CVector<TYPE>::Distance(const CVector<TYPE>& v) const
{
	return sqrt((X-v.X)*(X-v.X) + (Y-v.Y)*(Y-v.Y) + (Z-v.Z)*(Z-v.Z));
}


///////////////////////////////////////////////////////////////////////////
//// I/O support

template <class TYPE>
istream& operator>>(istream& is, CVector<TYPE>& v)
{
	TYPE tmp;

	is >> tmp; v.SetX(tmp);
	is >> tmp; v.SetY(tmp);
	is >> tmp; v.SetZ(tmp);

	return is;
}

template <class TYPE>
ostream& operator<<(ostream& os, CVector<TYPE>& v)
{
	os << v.GetX() << " " << v.GetY() << " " << v.GetZ();
	return os;
}

}

#endif
